/** 
 *  Copyright (c) 1999~2017, Altibase Corp. and/or its affiliates. All rights reserved.
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU Affero General Public License, version 3,
 *  as published by the Free Software Foundation.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *  GNU Affero General Public License for more details.
 *
 *  You should have received a copy of the GNU Affero General Public License
 *  along with this program. If not, see <http://www.gnu.org/licenses/>.
 */
 
package com.altibase.altilinker.session;

import java.sql.SQLException;

/**
 * Remote server error class by JDBC interface (by java.sql.Exception)
 */
public class RemoteServerError
{
    private String mSQLState;
    private int    mErrorCode;
    private String mErrorMessage;

    /**
     * Constructor
     * 
     * @param aSQLState         SQL state code generated by remote server
     * @param aErrorCode        Error code generated by remote server
     * @param aErrorMessage     Error message generated by remote server
     */
    public RemoteServerError(String aSQLState,
                             int    aErrorCode,
                             String aErrorMessage)
    {
        mSQLState     = aSQLState;
        mErrorCode    = aErrorCode;
        mErrorMessage = aErrorMessage;
    }

    /**
     * Constructor
     * 
     * @param e         java.sql.Exception
     */
    public RemoteServerError(SQLException e)
    {
        this(e.getSQLState(), e.getErrorCode(), e.getMessage());
    }
    
    /**
     * Get SQL state code of remote server error
     * 
     * @return  SQL state code
     */
    public String getSQLState()
    {
        return mSQLState;
    }
    
    /**
     * Get error code of remote server error
     * 
     * @return  Error code of remote server error
     */
    public int getErrorCode()
    {
        return mErrorCode;
    }
    
    /**
     * Get error message of remote server error
     * 
     * @return  Error message of remote server error
     */
    public String getErrorMessage()
    {
        return mErrorMessage;
    }
    
    /**
     * Whether remote server error is connection error or not
     * 
     * @return  true, if remote server error is connection error. false, if not.
     */
    public boolean isConnectionError()
    {
        return isConnectionError(mSQLState);
    }

    /**
     * Whether remote server error by SQL state code is connection error or not
     * 
     * @param aSQLState         SQL state code
     * @return                  true, if remote server error by SQL state code is connection error. false, if not.
     */
    static public boolean isConnectionError(String aSQLState)
    {
        if (aSQLState == null)
        {
            return false;
        }
        
        boolean sConnectionError = false;
        
        if ((aSQLState.equals("01002") == true) || // Disconnect error
            (aSQLState.equals("08000") == true) || // Connection exception
            (aSQLState.equals("08001") == true) || // Unable to connect to the data source, e.g. invalid license key
            (aSQLState.equals("08002") == true) || // Connection already in use
            (aSQLState.equals("08003") == true) || // Connection not open
            (aSQLState.equals("08004") == true) || // Data source rejected establishment of connection
            (aSQLState.equals("08007") == true) || // Connection failure during transaction
            (aSQLState.equals("08900") == true) || // Server lookup failed
            (aSQLState.equals("08S01") == true))   // Communication link failure
        {
            sConnectionError = true;
        }
        
        return sConnectionError;
    }
}
