package Altibase.jdbc.driver;

import java.io.*;
import java.sql.*;

import static org.junit.Assert.assertArrayEquals;

public class BlobTest extends AltibaseTestCase
{
    protected String[] getCleanQueries()
    {
        return new String[] {
                "DROP TABLE t1",
        };
    }

    protected String[] getInitQueries()
    {
        return new String[] {
                "CREATE TABLE t1 (id INTEGER, image BLOB)",
        };
    }

    protected String getURL()
    {
        return super.getURL() + "?AUTO_COMMIT=0";
    }

    private static final byte[] DAT_32_VAL1 = new byte[]{0x00,0x01,0x02,0x03,0x04,0x05,0x06,0x07,0x08,0x09,0x0a,0x0b,0x0c,0x0d,0x0e,0x0f,0x10,0x11,0x12,0x13,0x14,0x15,0x16,0x17,0x18,0x19,0x1a,0x1b,0x1c,0x1d,0x1e,0x1f};
    private static final byte[] DAT_32_VAL2 = new byte[]{0x20,0x21,0x22,0x23,0x24,0x25,0x26,0x27,0x28,0x29,0x2a,0x2b,0x2c,0x2d,0x2e,0x2f,0x30,0x31,0x32,0x33,0x34,0x35,0x36,0x37,0x38,0x39,0x3a,0x3b,0x3c,0x3d,0x3e,0x3f};

    private static final int TEST_TYPE_BY_BIN_STREAM1   = 1;
    private static final int TEST_TYPE_BY_OUTPARAM      = 2;
    private static final int TEST_TYPE_BY_BYTES         = 3;
    private static final int TEST_TYPE_BY_LONGVARBINARY = 4;
    private static final int TEST_TYPE_BY_BIN_STREAM2   = 5;
    private static final int TEST_TYPE_BY_OBJ_STREAM    = 6;

    public void testBlobByOutParam() throws SQLException, IOException
    {
        testBlobBy(TEST_TYPE_BY_OUTPARAM, true);
    }

    public void testBlobByStream1() throws SQLException, IOException
    {
        testBlobBy(TEST_TYPE_BY_BIN_STREAM1, true);
    }

    public void testBlobByStream2() throws SQLException, IOException
    {
        testBlobBy(TEST_TYPE_BY_BIN_STREAM2, true);
    }

    public void testBlobByBytesObject() throws SQLException, IOException
    {
        testBlobBy(TEST_TYPE_BY_LONGVARBINARY, true);
    }

    public void testBlobByStreamObject() throws SQLException, IOException
    {
        testBlobBy(TEST_TYPE_BY_OBJ_STREAM, true);
    }

    public void testBlobByBytes() throws SQLException, IOException
    {
        testBlobBy(TEST_TYPE_BY_BYTES, false);
    }

    public void testBlobBy(int aTestType, boolean aUseLargeBin) throws SQLException, IOException
    {
        assertBlob("test/resources/LargeGeometry.bin", 1001, aTestType);
        assertBlob("test/resources/dat_65534.dat", 1002, aTestType);
        if (aUseLargeBin)
        {
            assertBlob("test/resources/dat_64k.dat", 1003, aTestType);
            assertBlob("test/resources/dat_300k.dat", 1004, aTestType);
        }
        assertBlob("test/resources/dat_32.dat", 1005, aTestType);
    }
    
    private void assertBlob(String aInputBinaryFileName, int aFileId, int aTestType) throws SQLException, IOException
    {
        String sOutputBinaryFileNameByGetBytes = aInputBinaryFileName + ".getBytes.out";
        String sOutputBinaryFileNameByStream = aInputBinaryFileName + ".stream.out";

        File sInputBinaryFile = new File(aInputBinaryFileName);
        if (!sInputBinaryFile.exists())
        {
            throw new IOException("File not found. " + aInputBinaryFileName);
        }

        // Oracle ׳ setXXX() . CallableStatement  Ѵ.
        switch (aTestType)
        {
            case TEST_TYPE_BY_OUTPARAM:
                loadBlobFromFileByOutParam(connection(), aFileId, aInputBinaryFileName);
                break;
            default:
                loadBlobFromFileByValue(connection(), aFileId, aInputBinaryFileName, aTestType);
                break;
        }
        saveBlobToFileByBytes(connection(), aFileId, sOutputBinaryFileNameByGetBytes);
        saveBlobToFileByStream(connection(), aFileId, sOutputBinaryFileNameByStream);

        assertFileEquals(aInputBinaryFileName, sOutputBinaryFileNameByGetBytes);
        assertFileEquals(aInputBinaryFileName, sOutputBinaryFileNameByStream);

        deleteFile(sOutputBinaryFileNameByGetBytes);
        deleteFile(sOutputBinaryFileNameByStream);
    }

    private void deleteFile(String aPath)
    {
        new File(aPath).delete();
    }

    private void loadBlobFromFileByValue(Connection aConn, int aFileId, String aInFileName, int aTestType) throws SQLException, IOException
    {
        File aInFile = new File(aInFileName);
        FileInputStream sFileIn = new FileInputStream(aInFile);
        byte[] sBuf;

        PreparedStatement sStmt = aConn.prepareStatement("INSERT INTO t1 (id, image) VALUES (?, ?)");
        sStmt.setInt(1, aFileId);
        switch (aTestType)
        {
            case TEST_TYPE_BY_BIN_STREAM1:
                sStmt.setBinaryStream(2, sFileIn, (int)aInFile.length());
                break;
            case TEST_TYPE_BY_BIN_STREAM2:
                sStmt.setBinaryStream(2, sFileIn, Integer.MAX_VALUE);
                break;
            case TEST_TYPE_BY_OBJ_STREAM:
                sStmt.setObject(2, sFileIn);
                break;
            case TEST_TYPE_BY_LONGVARBINARY:
                sBuf = new byte[(int)aInFile.length()];
                assertEquals(sBuf.length, sFileIn.read(sBuf));
                sStmt.setObject(2, sBuf, Types.LONGVARBINARY);
                break;
            case TEST_TYPE_BY_BYTES:
                sBuf = new byte[(int)aInFile.length()];
                assertEquals(sBuf.length, sFileIn.read(sBuf));
                sStmt.setBytes(2, sBuf);
                break;
        }
        assertEquals(1, sStmt.executeUpdate());

        sFileIn.close();
        aConn.commit();
        sStmt.close();
    }

    private void loadBlobFromFileByOutParam(Connection aConn, int aFileId, String aInFileName) throws SQLException, IOException
    {
        File aInFile = new File(aInFileName);
        FileInputStream sFileIn = new FileInputStream(aInFile);
        byte[] sBuf = new byte[(int)aInFile.length()];
        assertEquals(sBuf.length, sFileIn.read(sBuf));
        sFileIn.close();

        CallableStatement sCallStmt = aConn.prepareCall("INSERT INTO t1 (id, image) VALUES (?, ?)");
        sCallStmt.setInt(1, aFileId);
        sCallStmt.registerOutParameter(2, Types.BLOB);
        sCallStmt.execute();

        Blob sBlob = sCallStmt.getBlob(2);
        OutputStream sBlobOut = sBlob.setBinaryStream(0);
        sBlobOut.write(sBuf);
        sBlobOut.close();
        
        aConn.commit();
        sCallStmt.close();
    }

    private void loadBlob(int aId, byte[] aValue) throws SQLException
    {
        PreparedStatement sInsStmt = connection().prepareStatement("INSERT INTO t1 VALUES (?, ?)");
        sInsStmt.setInt(1, aId);
        sInsStmt.setBinaryStream(2, new ByteArrayInputStream(aValue), aValue.length);
        assertEquals(1, sInsStmt.executeUpdate());
        sInsStmt.close();
    }

    private void saveBlobToFileByBytes(Connection aConn, int aFileId, String aOutFileName) throws SQLException, IOException
    {
        Statement sStmt = aConn.createStatement();
        ResultSet sRS = sStmt.executeQuery("SELECT image FROM t1 WHERE id = " + aFileId);
        assertEquals(true, sRS.next());

        byte[] sBuf = sRS.getBytes(1);
        File sOutFile = new File(aOutFileName);
        if (sOutFile.exists())
        {
            sOutFile.delete();
        }
        FileOutputStream sFileOut = new FileOutputStream(sOutFile);
        sFileOut.write(sBuf, 0, sBuf.length);
        sFileOut.close();

        sRS.close();
        sStmt.close();
    }

    private void saveBlobToFileByStream(Connection aConn, int aFileId, String aOutFileName) throws SQLException, IOException
    {
        Statement sStmt = aConn.createStatement();
        ResultSet sRS = sStmt.executeQuery("SELECT image FROM t1 WHERE id = " + aFileId);
        assertEquals(true, sRS.next());
        Blob sBlob = sRS.getBlob(1);

        InputStream sBlobIn = sBlob.getBinaryStream();
        File sOutFile = new File(aOutFileName);
        if (sOutFile.exists())
        {
            sOutFile.delete();
        }
        FileOutputStream sFileOut = new FileOutputStream(sOutFile);
        byte[] sBuf = new byte[(int)sBlob.length()];
        assertEquals(sBuf.length, sBlobIn.read(sBuf));
        sFileOut.write(sBuf);
        sFileOut.close();
        sBlobIn.close();

        sRS.close();
        sStmt.close();
    }

    private void assertFileEquals(String aExpFilePath, String aActFilePath) throws IOException
    {
        File sFile1 = new File(aExpFilePath);
        File sFile2 = new File(aActFilePath);

        assertEquals(sFile1.length(), sFile2.length());

        FileInputStream sIn1 = new FileInputStream(sFile1);
        FileInputStream sIn2 = new FileInputStream(sFile2);

        byte[] sBuf1 = new byte[(int)sFile1.length()];
        byte[] sBuf2 = new byte[(int)sFile2.length()];

        assertEquals(sBuf1.length, sIn1.read(sBuf1));
        assertEquals(sBuf2.length, sIn2.read(sBuf2));

        sIn1.close();
        sIn2.close();

        assertArrayEquals(sBuf1, sBuf2);
    }

    public void testGetBytesFromBlob() throws SQLException, IOException
    {
        loadBlobFromFileByValue(connection(), 1001, "test/resources/dat_32.dat", TEST_TYPE_BY_BYTES);

        Statement sStmt = connection().createStatement();
        ResultSet sRS = sStmt.executeQuery("SELECT image FROM t1 WHERE id = 1001");
        assertEquals(true, sRS.next());
        Blob sBlob = sRS.getBlob(1);
        assertEquals(32, sBlob.length());
        byte[] sBytes1 = sBlob.getBytes(1, 16);
        assertArrayEquals(new byte[]{0x00,0x01,0x02,0x03,0x04,0x05,0x06,0x07,0x08,0x09,0x0a,0x0b,0x0c,0x0d,0x0e,0x0f}, sBytes1);
        byte[] sBytes2 = sBlob.getBytes(17, 16);
        assertArrayEquals(new byte[]{0x10,0x11,0x12,0x13,0x14,0x15,0x16,0x17,0x18,0x19,0x1a,0x1b,0x1c,0x1d,0x1e,0x1f}, sBytes2);
        assertEquals(false, sRS.next());
    }

    public void testGetBytesFromBlob2() throws Exception
    {
        loadBlob(1, DAT_32_VAL1);

        Statement sStmt = connection().createStatement();
        ResultSet sRS = sStmt.executeQuery("SELECT * FROM t1");
        assertTrue(sRS.next());
        assertEquals(1, sRS.getInt(1));
        Object sObj = sRS.getObject(2);
        assertTrue(sObj instanceof Blob);
        Blob sBlob = (Blob)sObj;
        assertEquals(DAT_32_VAL1.length, sBlob.length());
        assertArrayEquals(DAT_32_VAL1, sBlob.getBytes(1, (int)sBlob.length()));
        assertEquals(false, sRS.next());
        sRS.close();
        sStmt.close();
    }

    public void testSetBLOB() throws SQLException, IOException
    {
        loadBlobFromFileByValue(connection(), 1, "test/resources/dat_32.dat", TEST_TYPE_BY_BYTES);

        Statement sStmt = connection().createStatement();
        ResultSet sRS = sStmt.executeQuery("SELECT * FROM t1");
        assertEquals(true, sRS.next());
        assertEquals(1, sRS.getInt(1));
        Blob sBlob = sRS.getBlob(2);
        assertArrayEquals(DAT_32_VAL1, sBlob.getBytes(1, 32));
        assertEquals(false, sRS.next());
        sStmt.close();

        PreparedStatement sInsStmt = connection().prepareStatement("INSERT INTO t1 VALUES (?, ?)");
        sInsStmt.setInt(1, 2);
        sInsStmt.setObject(2, sBlob, Types.BLOB);
        assertEquals(false, sInsStmt.execute());
        sInsStmt.close();
        
        sStmt = connection().createStatement();
        sRS = sStmt.executeQuery("SELECT * FROM t1");
        assertEquals(true, sRS.next());
        assertEquals(1, sRS.getInt(1));
        assertArrayEquals(DAT_32_VAL1, sRS.getBytes(2));
        assertEquals(true, sRS.next());
        assertEquals(2, sRS.getInt(1));
        assertArrayEquals(DAT_32_VAL1, sRS.getBytes(2));
        assertEquals(false, sRS.next());
        sStmt.close();
    }

    public void testUpdateObjectByBlob() throws SQLException
    {
        loadBlob(1, DAT_32_VAL1);
        loadBlob(2, DAT_32_VAL2);

        // 2° row blob 1° blob  update
        Statement sUpdStmt = connection().createStatement(ResultSet.TYPE_FORWARD_ONLY, ResultSet.CONCUR_UPDATABLE, ResultSet.CLOSE_CURSORS_AT_COMMIT);
        ResultSet sUpdRS = sUpdStmt.executeQuery("SELECT id, image FROM t1");
        assertEquals(true, sUpdRS.next());
        assertEquals(1, sUpdRS.getInt(1));
        assertArrayEquals(DAT_32_VAL1, sUpdRS.getBytes(2));
        Blob sBlob = sUpdRS.getBlob(2);
        assertEquals(true, sUpdRS.next());
        assertEquals(2, sUpdRS.getInt(1));
        assertArrayEquals(DAT_32_VAL2, sUpdRS.getBytes(2));
        sUpdRS.updateObject(2, sBlob);
        sUpdRS.updateRow();
        assertEquals(false, sUpdRS.next());
        sUpdStmt.close();

        Statement sSelStmt = connection().createStatement();
        ResultSet sSelRS = sSelStmt.executeQuery("SELECT * FROM t1");
        assertEquals(true, sSelRS.next());
        assertEquals(1, sSelRS.getInt(1));
        assertArrayEquals(DAT_32_VAL1, sSelRS.getBytes(2));
        assertEquals(true, sSelRS.next());
        assertEquals(2, sSelRS.getInt(1));
        assertArrayEquals(DAT_32_VAL1, sSelRS.getBytes(2));
        assertEquals(false, sSelRS.next());
        sSelStmt.close();
    }
    
    public void testUpdateObjectByBinaryStream() throws SQLException, IOException
    {
        loadBlob(1, DAT_32_VAL1);

        Statement sUpdStmt = connection().createStatement(ResultSet.TYPE_FORWARD_ONLY, ResultSet.CONCUR_UPDATABLE, ResultSet.CLOSE_CURSORS_AT_COMMIT);
        ResultSet sUpdRS = sUpdStmt.executeQuery("SELECT id, image FROM t1");
        assertEquals(true, sUpdRS.next());
        assertEquals(1, sUpdRS.getInt(1));
        assertArrayEquals(DAT_32_VAL1, sUpdRS.getBytes(2));
        InputStream sBinIN2 = new ByteArrayInputStream(DAT_32_VAL2);
        try {
            sUpdRS.updateObject(2, sBinIN2);
            fail();
        }
        catch(SQLException e)
        {
            
        }
        finally
        {
            sUpdStmt.close();
        }
    }

    public void testUpdateObjectByByteArr() throws SQLException, IOException
    {
        loadBlob(1, DAT_32_VAL1);

        Statement sUpdStmt = connection().createStatement(ResultSet.TYPE_FORWARD_ONLY, ResultSet.CONCUR_UPDATABLE, ResultSet.CLOSE_CURSORS_AT_COMMIT);
        ResultSet sUpdRS = sUpdStmt.executeQuery("SELECT id, image FROM t1");
        assertEquals(ResultSet.CONCUR_UPDATABLE, sUpdRS.getConcurrency());
        assertEquals(true, sUpdRS.next());
        assertEquals(1, sUpdRS.getInt(1));
        assertArrayEquals(DAT_32_VAL1, sUpdRS.getBytes(2));
        sUpdRS.updateObject(2, DAT_32_VAL2);
        sUpdRS.updateRow();
        assertEquals(false, sUpdRS.next());
        sUpdStmt.close();

        Statement sSelStmt = connection().createStatement();
        ResultSet sSelRS = sSelStmt.executeQuery("SELECT * FROM t1");
        assertEquals(true, sSelRS.next());
        assertEquals(1, sSelRS.getInt(1));
        assertArrayEquals(DAT_32_VAL2, sSelRS.getBytes(2));
        assertEquals(false, sSelRS.next());
        sSelStmt.close();
    }

    public void testUpdateBlob() throws SQLException
    {
        loadBlob(1, DAT_32_VAL1);
        loadBlob(2, DAT_32_VAL2);

        // 2° row blob 1° blob  update
        Statement sUpdStmt = connection().createStatement(ResultSet.TYPE_FORWARD_ONLY, ResultSet.CONCUR_UPDATABLE, ResultSet.CLOSE_CURSORS_AT_COMMIT);
        ResultSet sUpdRS = sUpdStmt.executeQuery("SELECT id, image FROM t1");
        assertEquals(true, sUpdRS.next());
        assertEquals(1, sUpdRS.getInt(1));
        assertArrayEquals(DAT_32_VAL1, sUpdRS.getBytes(2));
        Blob sBlob = sUpdRS.getBlob(2);
        assertEquals(true, sUpdRS.next());
        assertEquals(2, sUpdRS.getInt(1));
        assertArrayEquals(DAT_32_VAL2, sUpdRS.getBytes(2));
        sUpdRS.updateBlob(2, sBlob);
        sUpdRS.updateRow();
        assertEquals(false, sUpdRS.next());
        sUpdStmt.close();

        Statement sSelStmt = connection().createStatement();
        ResultSet sSelRS = sSelStmt.executeQuery("SELECT * FROM t1");
        assertEquals(true, sSelRS.next());
        assertEquals(1, sSelRS.getInt(1));
        assertArrayEquals(DAT_32_VAL1, sSelRS.getBytes(2));
        assertEquals(true, sSelRS.next());
        assertEquals(2, sSelRS.getInt(1));
        assertArrayEquals(DAT_32_VAL1, sSelRS.getBytes(2));
        assertEquals(false, sSelRS.next());
        sSelStmt.close();
    }

    public void testUpdateBlobByBinaryStream() throws SQLException, IOException
    {
        loadBlob(1, DAT_32_VAL1);

        Statement sUpdStmt = connection().createStatement(ResultSet.TYPE_FORWARD_ONLY, ResultSet.CONCUR_UPDATABLE, ResultSet.CLOSE_CURSORS_AT_COMMIT);
        ResultSet sUpdRS = sUpdStmt.executeQuery("SELECT id, image FROM t1");
        assertEquals(true, sUpdRS.next());
        assertEquals(1, sUpdRS.getInt(1));
        assertArrayEquals(DAT_32_VAL1, sUpdRS.getBytes(2));
        InputStream sBinIN2 = new ByteArrayInputStream(DAT_32_VAL2);
        sUpdRS.updateBinaryStream(2, sBinIN2, sBinIN2.available());
        sUpdRS.updateRow();
        assertEquals(false, sUpdRS.next());
        sUpdStmt.close();

        Statement sSelStmt = connection().createStatement();
        ResultSet sSelRS = sSelStmt.executeQuery("SELECT * FROM t1");
        assertEquals(true, sSelRS.next());
        assertEquals(1, sSelRS.getInt(1));
        assertArrayEquals(DAT_32_VAL2, sSelRS.getBytes(2));
        assertEquals(false, sSelRS.next());
        sSelStmt.close();
    }

    public void testSetNullByNULL() throws SQLException
    {
        testSetNullBy(Types.NULL);
    }

    public void testSetNullByBLOB() throws SQLException
    {
        testSetNullBy(Types.BLOB);
    }

    public void testSetNullByBINARY() throws SQLException
    {
        testSetNullBy(Types.BINARY);
    }

    public void testSetNullByVARBINARY() throws SQLException
    {
        testSetNullBy(Types.VARBINARY);
    }

    public void testSetNullByLONGVARBINARY() throws SQLException
    {
        testSetNullBy(Types.LONGVARBINARY);
    }

    public void testSetNullByBinaryStream() throws SQLException
    {
        testSetNullBy(TYPES_BINARY_STREAM);
    }

    private static final int TYPES_ASCII_STREAM  = -100001;
    private static final int TYPES_BINARY_STREAM = -100002;
    private static final int TYPES_CHAR_STREAM   = -100003;

    private void testSetNullBy(int aJDBCType) throws SQLException
    {
        PreparedStatement sInsStmt = connection().prepareStatement("INSERT INTO t1 VALUES (?, ?)");
        sInsStmt.setInt(1, 1);
        switch (aJDBCType)
        {
            case TYPES_ASCII_STREAM:
                sInsStmt.setAsciiStream(2, null, 0);
                break;
            case TYPES_BINARY_STREAM:
                sInsStmt.setBinaryStream(2, null, 0);
                break;
            case TYPES_CHAR_STREAM:
                sInsStmt.setCharacterStream(2,  null, 0);
                break;
            default:
                sInsStmt.setObject(2, null, aJDBCType);
                break;
        }
        assertEquals(1, sInsStmt.executeUpdate());
        sInsStmt.close();

        Statement sSelStmt = connection().createStatement();
        ResultSet sRS = sSelStmt.executeQuery("SELECT * FROM t1");
        assertEquals(true, sRS.next());
        assertEquals(1, sRS.getInt(1));
        assertEquals(null, sRS.getObject(2));
        assertEquals(false, sRS.next());
        sSelStmt.close();
    }

    public void testLobTruncate() throws Exception
    {
        loadBlob(1, DAT_32_VAL1);

        Statement sStmt = connection().createStatement();
        ResultSet sRS = sStmt.executeQuery("SELECT * FROM t1 FOR UPDATE");
        assertEquals(true, sRS.next());
        Blob sBlob = sRS.getBlob(2);
        byte[] sTruncatedVal = new byte[DAT_32_VAL1.length / 2];
        System.arraycopy(DAT_32_VAL1, 0, sTruncatedVal, 0, sTruncatedVal.length);
        sBlob.truncate(sTruncatedVal.length);
        assertEquals(sTruncatedVal.length, sBlob.length());
        assertArrayEquals(sTruncatedVal, sBlob.getBytes(1, (int)sBlob.length()));
        assertEquals(false, sRS.next());
        sRS.close();
        sStmt.close();
    }

    public void testSetBinaryStream() throws SQLException, IOException
    {
        testSetBinaryStream(false);
    }

    public void testSetBinaryStreamAsObject() throws SQLException, IOException
    {
        testSetBinaryStream(true);
    }

    public void testSetBinaryStream(boolean aUsingSetObject) throws SQLException, IOException
    {
        InputStream sBinIN1 = new ByteArrayInputStream(DAT_32_VAL1);
        InputStream sBinIN2 = new ByteArrayInputStream(DAT_32_VAL2);

        PreparedStatement sInsStmt = connection().prepareStatement("INSERT INTO t1 VALUES (?, ?)");
        sInsStmt.setInt(1, 1);
        if (aUsingSetObject == true)
        {
            sInsStmt.setObject(2, sBinIN1);
        }
        else
        {
            sInsStmt.setBinaryStream(2, sBinIN1, sBinIN1.available());
        }
        assertEquals(false, sInsStmt.execute());
        sInsStmt.setInt(1, 2);
        if (aUsingSetObject == true)
        {
            sInsStmt.setObject(2, sBinIN2);
        }
        else
        {
            sInsStmt.setBinaryStream(2, sBinIN2, sBinIN2.available());
        }
        assertEquals(false, sInsStmt.execute());
        sInsStmt.close();

        Statement sStmt = connection().createStatement();
        ResultSet sSelRS = sStmt.executeQuery("SELECT * FROM t1");
        assertEquals(true, sSelRS.next());
        assertEquals(1, sSelRS.getInt(1));
        assertArrayEquals(DAT_32_VAL1, sSelRS.getBytes(2));
        assertEquals(true, sSelRS.next());
        assertEquals(2, sSelRS.getInt(1));
        assertArrayEquals(DAT_32_VAL2, sSelRS.getBytes(2));
        assertEquals(false, sSelRS.next());
        sStmt.close();
    }
}
